<?php
// models/Studio.php

// Pastikan BaseModel.php di-include. Sesuaikan path jika struktur direktori Anda berbeda.
require_once dirname(__DIR__) . '/models/BaseModel.php';

class Studio extends BaseModel {
    // Properti yang mendefinisikan nama tabel di database
    protected $table = 'studio';

    /**
     * Mengambil semua data studio dari database.
     *
     * @return array Array asosiatif berisi semua data studio.
     */
    public function getAllStudios() {
        $sql = "SELECT * FROM {$this->table} ORDER BY nama ASC"; // Urutkan berdasarkan nama studio
        $result = $this->conn->query($sql);
        $studios = [];
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $studios[] = $row;
            }
        }
        return $studios;
    }

    /**
     * Mengambil data studio berdasarkan ID.
     *
     * @param int $id ID studio.
     * @return array|null Array asosiatif data studio jika ditemukan, null jika tidak.
     */
    public function getStudioById($id) {
        $stmt = $this->conn->prepare("SELECT * FROM {$this->table} WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }

    /**
     * Menambahkan studio baru ke database.
     *
     * @param string $nama Nama studio.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function addStudio($nama) {
        $stmt = $this->conn->prepare("INSERT INTO {$this->table} (nama) VALUES (?)");
        $stmt->bind_param("s", $nama);
        return $stmt->execute();
    }

    /**
     * Memperbarui data studio yang sudah ada di database.
     *
     * @param int $id ID studio yang akan diperbarui.
     * @param string $nama Nama studio baru.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function updateStudio($id, $nama) {
        $stmt = $this->conn->prepare("UPDATE {$this->table} SET nama = ? WHERE id = ?");
        $stmt->bind_param("si", $nama, $id);
        return $stmt->execute();
    }

    /**
     * Menghapus studio dari database berdasarkan ID.
     * Perhatikan FOREIGN KEY CONSTRAINTS. Jika ada film atau kursi terkait studio ini,
     * penghapusan mungkin akan diblokir.
     *
     * @param int $id ID studio yang akan dihapus.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function deleteStudio($id) {
        $stmt = $this->conn->prepare("DELETE FROM {$this->table} WHERE id = ?");
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }

    /**
     * Menghitung total jumlah baris di tabel studio.
     * @return int Jumlah total studio.
     */
    public function countAll() {
        $result = $this->conn->query("SELECT COUNT(*) FROM {$this->table}");
        if (!$result) {
            error_log("SQL Error in Studio::countAll: " . $this->conn->error);
            return 0;
        }
        $row = $result->fetch_row();
        return $row[0];
    }
}
