<?php
// models/Kursi.php

// Pastikan BaseModel.php di-include. Sesuaikan path jika struktur direktori Anda berbeda.
require_once dirname(__DIR__) . '/models/BaseModel.php';

class Kursi extends BaseModel {
    // Properti yang mendefinisikan nama tabel di database
    protected $table = 'kursi';

    /**
     * Mengambil semua data kursi dari database.
     * Termasuk informasi studio melalui JOIN.
     *
     * @return array Array asosiatif berisi semua data kursi.
     */
    public function getAllKursi() {
        $sql = "SELECT k.*, s.nama AS studio_nama
                FROM {$this->table} k
                JOIN studio s ON k.studio_id = s.id
                ORDER BY s.nama ASC, k.nomor ASC"; // Urutkan berdasarkan studio dan nomor kursi
        $result = $this->conn->query($sql);
        $kursis = [];
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $kursis[] = $row;
            }
        }
        return $kursis;
    }

    /**
     * Mengambil data kursi berdasarkan ID.
     * Termasuk informasi studio.
     *
     * @param int $id ID kursi.
     * @return array|null Array asosiatif data kursi jika ditemukan, null jika tidak.
     */
    public function getKursiById($id) {
        $stmt = $this->conn->prepare("SELECT k.*, s.nama AS studio_nama
                                     FROM {$this->table} k
                                     JOIN studio s ON k.studio_id = s.id
                                     WHERE k.id = ?");
        // 'i' menunjukkan bahwa parameter adalah integer
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }

    /**
     * Mengambil semua kursi untuk studio tertentu.
     *
     * @param int $studio_id ID studio.
     * @return array Array asosiatif berisi data kursi untuk studio tersebut.
     */
    public function getKursiByStudioId($studio_id) {
        $stmt = $this->conn->prepare("SELECT * FROM {$this->table} WHERE studio_id = ? ORDER BY nomor ASC");
        $stmt->bind_param("i", $studio_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $kursis = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $kursis[] = $row;
            }
        }
        return $kursis;
    }

    /**
     * Menambahkan kursi baru ke database.
     *
     * @param int $studio_id ID studio tempat kursi berada.
     * @param string $nomor Nomor kursi (misal: A1, B10).
     * @return bool True jika berhasil, false jika gagal.
     */
    public function addKursi($studio_id, $nomor) {
        $stmt = $this->conn->prepare("INSERT INTO {$this->table} (studio_id, nomor) VALUES (?, ?)");
        // 'is' menunjukkan tipe parameter: integer, string
        $stmt->bind_param("is", $studio_id, $nomor);
        return $stmt->execute();
    }

    /**
     * Memperbarui data kursi yang sudah ada di database.
     *
     * @param int $id ID kursi yang akan diperbarui.
     * @param int $studio_id ID studio tempat kursi berada.
     * @param string $nomor Nomor kursi baru.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function updateKursi($id, $studio_id, $nomor) {
        $stmt = $this->conn->prepare("UPDATE {$this->table} SET studio_id = ?, nomor = ? WHERE id = ?");
        // 'isi' menunjukkan tipe parameter: integer, string, integer
        $stmt->bind_param("isi", $studio_id, $nomor, $id);
        return $stmt->execute();
    }

    /**
     * Menghapus kursi dari database berdasarkan ID.
     * Perhatikan FOREIGN KEY CONSTRAINTS. Jika ada pemesanan terkait kursi ini (di `detail_pemesanan`),
     * penghapusan mungkin akan diblokir.
     *
     * @param int $id ID kursi yang akan dihapus.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function deleteKursi($id) {
        $stmt = $this->conn->prepare("DELETE FROM {$this->table} WHERE id = ?");
        // 'i' menunjukkan bahwa parameter adalah integer
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }
}
