<?php
// controllers/StudioController.php

// Pastikan semua model dan BaseController di-include dengan jalur yang benar
require_once dirname(__DIR__) . '/models/Studio.php'; // Model Studio
require_once dirname(__DIR__) . '/controllers/BaseController.php'; // BaseController
// Pastikan StudioController menggunakan Detail_Pemesanan jika dibutuhkan (misal di destroy)
require_once dirname(__DIR__) . '/models/Detail_Pemesanan.php'; // Diperlukan untuk pengecekan destroy

// Pastikan kelas ini mewarisi dari BaseController
class StudioController extends BaseController {

    /**
     * Konstruktor untuk StudioController.
     * Menerima objek koneksi MySQLi dan meneruskannya ke BaseController.
     * @param mysqli $conn Objek koneksi database MySQLi.
     */
    public function __construct(mysqli $conn) {
        parent::__construct($conn);
    }

    /**
     * Menampilkan daftar semua studio.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=studio
     */
    public function index() {
        // Memerlukan peran admin
        $this->requireAdmin();

        $studioModel = new Studio($this->conn); // Meneruskan koneksi
        $studios = $studioModel->getAllStudios();

        // Load view untuk menampilkan daftar studio
        $this->view('studio', ['studios' => $studios]);
    }

    /**
     * Menampilkan formulir untuk menambah studio baru.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=studio_create
     */
    public function create() {
        $this->requireAdmin();

        $this->view('studio_form'); // views/studio_form.php untuk tambah studio
    }

    /**
     * Memproses data POST dari formulir penambahan studio baru.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=studio_store
     */
    public function store() {
        $this->requireAdmin();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $nama = $_POST['nama'] ?? '';

            if (empty($nama)) {
                $this->setErrorMessage("Nama studio harus diisi.");
                $this->redirect('studio_create');
                exit();
            }

            $studioModel = new Studio($this->conn); // Meneruskan koneksi
            $result = $studioModel->addStudio($nama);

            if ($result) {
                $this->setSuccessMessage("Studio berhasil ditambahkan.");
                $this->redirect('studio'); // Redirect ke daftar studio
                exit();
            } else {
                $this->setErrorMessage("Gagal menambahkan studio. Nama studio mungkin sudah ada.");
                $this->redirect('studio_create'); // Kembali ke form
                exit();
            }
        } else {
            $this->redirect('studio_create');
            exit();
        }
    }

    /**
     * Menampilkan formulir untuk mengedit studio yang sudah ada.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=studio_edit&id=X
     */
    public function edit($studio_id) {
        $this->requireAdmin();

        $studioModel = new Studio($this->conn); // Meneruskan koneksi
        $studio = $studioModel->getStudioById($studio_id);

        if (!$studio) {
            $this->setErrorMessage("Studio tidak ditemukan.");
            $this->redirect('studio');
            exit();
        }

        $this->view('studio_form_update', ['studio' => $studio]);
    }

    /**
     * Memproses data POST dari formulir pembaruan studio.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=studio_update
     */
    public function update() {
        $this->requireAdmin();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $id = $_POST['id'] ?? null;
            $nama = $_POST['nama'] ?? '';

            if (empty($id) || empty($nama)) {
                $this->setErrorMessage("Data tidak lengkap untuk pembaruan studio.");
                $this->redirect('studio_edit', ['id' => $id]);
                exit();
            }

            $studioModel = new Studio($this->conn); // Meneruskan koneksi
            $result = $studioModel->updateStudio($id, $nama);

            if ($result) {
                $this->setSuccessMessage("Studio berhasil diperbarui.");
                $this->redirect('studio');
                exit();
            } else {
                $this->setErrorMessage("Gagal memperbarui studio. Nama studio mungkin sudah ada.");
                $this->redirect('studio_edit', ['id' => $id]);
                exit();
            }
        } else {
            $this->redirect('studio');
            exit();
        }
    }

    /**
     * Menghapus studio dari database.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=studio_destroy&id=X
     */
    public function destroy($studio_id) {
        $this->requireAdmin();

        $studioModel = new Studio($this->conn); // Meneruskan koneksi
        $detailPemesananModel = new Detail_Pemesanan($this->conn); // Meneruskan koneksi

        // Cek apakah ada kursi terkait studio ini
        $kursiModel = new Kursi($this->conn); // Diperlukan untuk mengecek kursi
        if (count($kursiModel->getKursiByStudioId($studio_id)) > 0) {
            $this->setErrorMessage("Tidak bisa menghapus studio. Ada kursi terkait studio ini.");
            $this->redirect('studio');
            exit();
        }

        // Cek apakah ada jadwal terkait studio ini
        // Mengambil jadwal yang terhubung ke studio ini melalui FILM
        $filmModel = new Film($this->conn); // Diperlukan untuk mengecek film
        $films_in_studio = $filmModel->getFilmsByStudioId($studio_id); // Asumsi ada method ini di FilmModel
        $has_related_jadwal = false;
        if (!empty($films_in_studio)) {
            $jadwalModel = new Jadwal($this->conn); // Diperlukan untuk mengecek jadwal
            foreach($films_in_studio as $film) {
                if (count($jadwalModel->getJadwalByFilmId($film['id'])) > 0) {
                    $has_related_jadwal = true;
                    break;
                }
            }
        }

        if ($has_related_jadwal) {
            $this->setErrorMessage("Tidak bisa menghapus studio. Ada film atau jadwal yang terkait dengan studio ini. Harap hapus film/jadwal tersebut terlebih dahulu.");
            $this->redirect('studio');
            exit();
        }

        $result = $studioModel->deleteStudio($studio_id);

        if ($result) {
            $this->setSuccessMessage("Studio berhasil dihapus.");
        } else {
            $this->setErrorMessage("Gagal menghapus studio. Kendala database.");
        }

        $this->redirect('studio');
        exit();
    }
}
