<?php
// controllers/KursiController.php

// Pastikan semua model yang relevan di-include
require_once dirname(__DIR__) . '/models/Kursi.php';
require_once dirname(__DIR__) . '/models/Studio.php'; // Perlu untuk memilih studio saat menambah/mengedit kursi
require_once dirname(__DIR__) . '/controllers/BaseController.php'; // Pastikan BaseController di-include

// Pastikan kelas ini mewarisi dari BaseController
class KursiController extends BaseController {

    /**
     * Konstruktor untuk KursiController.
     * Menerima objek koneksi MySQLi dan meneruskannya ke BaseController.
     * @param mysqli $conn Objek koneksi database MySQLi.
     */
    public function __construct(mysqli $conn) {
        // Panggil konstruktor parent (BaseController) untuk menginisialisasi $this->conn
        parent::__construct($conn);
    }

    /**
     * Menampilkan daftar semua kursi.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kursi
     */
    public function index() {
        // Memerlukan peran admin
        $this->requireAdmin();

        // BARIS 21: PASTIKAN INI ADALAH "$this->conn"
        $kursiModel = new Kursi($this->conn); 
        $kursis = $kursiModel->getAllKursi();

        // Load view untuk menampilkan daftar kursi
        $this->view('kursi', ['kursis' => $kursis]);
    }

    /**
     * Menampilkan formulir untuk menambah kursi baru.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kursi_create
     */
    public function create() {
        $this->requireAdmin();

        $studioModel = new Studio($this->conn); // Meneruskan koneksi
        $studios = $studioModel->getAllStudios();

        $this->view('kursi_form', ['studios' => $studios]);
    }

    /**
     * Memproses data POST dari formulir penambahan kursi baru.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kursi_store
     */
    public function store() {
        $this->requireAdmin();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $studio_id = $_POST['studio_id'] ?? '';
            $nomor = $_POST['nomor'] ?? '';

            if (empty($studio_id) || empty($nomor)) {
                $this->setErrorMessage("Studio dan Nomor Kursi harus diisi.");
                $this->redirect('kursi_create');
                exit();
            }

            $kursiModel = new Kursi($this->conn); // Meneruskan koneksi
            $result = $kursiModel->addKursi($studio_id, $nomor);

            if ($result) {
                $this->setSuccessMessage("Kursi berhasil ditambahkan.");
                $this->redirect('kursi'); // Redirect ke daftar kursi
                exit();
            } else {
                $this->setErrorMessage("Gagal menambahkan kursi. Nomor kursi mungkin sudah ada di studio ini.");
                $this->redirect('kursi_create'); // Kembali ke form
                exit();
            }
        } else {
            // Jika bukan POST request, redirect ke halaman tambah kursi
            $this->redirect('kursi_create');
            exit();
        }
    }

    /**
     * Menampilkan formulir untuk mengedit kursi yang sudah ada.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kursi_edit&id=X
     */
    public function edit($kursi_id) {
        $this->requireAdmin();

        $kursiModel = new Kursi($this->conn); // Meneruskan koneksi
        $studioModel = new Studio($this->conn); // Meneruskan koneksi

        $kursi = $kursiModel->getKursiById($kursi_id);
        $studios = $studioModel->getAllStudios();

        if (!$kursi) {
            $this->setErrorMessage("Kursi tidak ditemukan.");
            $this->redirect('kursi');
            exit();
        }

        $this->view('kursi_form_update', ['kursi' => $kursi, 'studios' => $studios]);
    }

    /**
     * Memproses data POST dari formulir pembaruan kursi.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kursi_update
     */
    public function update() {
        $this->requireAdmin();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $id = $_POST['id'] ?? null;
            $studio_id = $_POST['studio_id'] ?? '';
            $nomor = $_POST['nomor'] ?? '';

            if (empty($id) || empty($studio_id) || empty($nomor)) {
                $this->setErrorMessage("Data tidak lengkap untuk pembaruan kursi.");
                $this->redirect('kursi_edit', ['id' => $id]);
                exit();
            }

            $kursiModel = new Kursi($this->conn); // Meneruskan koneksi
            $result = $kursiModel->updateKursi($id, $studio_id, $nomor);

            if ($result) {
                $this->setSuccessMessage("Kursi berhasil diperbarui.");
                $this->redirect('kursi');
                exit();
            } else {
                $this->setErrorMessage("Gagal memperbarui kursi. Nomor kursi mungkin sudah ada di studio ini.");
                $this->redirect('kursi_edit', ['id' => $id]);
                exit();
            }
        } else {
            $this->redirect('kursi');
            exit();
        }
    }

    /**
     * Menghapus kursi dari database.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kursi_destroy&id=X
     */
    public function destroy($kursi_id) {
        $this->requireAdmin();

        $kursiModel = new Kursi($this->conn); // Meneruskan koneksi
        $detailPemesananModel = new Detail_Pemesanan($this->conn); // Diperlukan untuk pengecekan pemesanan
        
        // Cek apakah ada pemesanan terkait kursi ini sebelum menghapus
        if ($detailPemesananModel->getPemesananCountByKursi($kursi_id) > 0) { // Asumsi ada method getPemesananCountByKursi
            $this->setErrorMessage("Tidak bisa menghapus kursi. Ada pemesanan terkait kursi ini.");
            $this->redirect('kursi');
            exit();
        }

        $result = $kursiModel->deleteKursi($kursi_id);

        if ($result) {
            $this->setSuccessMessage("Kursi berhasil dihapus.");
        } else {
            $this->setErrorMessage("Gagal menghapus kursi. Kendala database.");
        }

        $this->redirect('kursi');
    }
}
