<?php
// controllers/FilmController.php

require_once __DIR__ . '/../controllers/BaseController.php';
require_once __DIR__ . '/../models/Film.php';
require_once __DIR__ . '/../models/Jadwal.php';
require_once __DIR__ . '/../models/Kategori.php';
require_once __DIR__ . '/../models/Studio.php';
require_once __DIR__ . '/../models/Users.php'; // Diperlukan untuk menghitung jumlah user
require_once __DIR__ . '/../models/Detail_Pemesanan.php'; // Diperlukan untuk pengecekan pemesanan terkait

class FilmController extends BaseController {

    public function __construct(mysqli $conn) {
        parent::__construct($conn);
    }

    /**
     * Menampilkan halaman utama (homepage) dengan carousel, featured films, counts, etc.
     * Dipanggil oleh index.php?hal=home
     */
    public function index() {
        $filmModel = new Film($this->conn);
        $userModel = new Users($this->conn);
        $studioModel = new Studio($this->conn);
        $kategoriModel = new Kategori($this->conn);

        // Ambil film untuk tampilan utama (featured films)
        // Jika getAllFilms() di Film.php sudah dioptimasi untuk ini, gunakan itu.
        // Jika perlu film tertentu (misal: hanya 5 film terbaru), buat method baru di FilmModel.
        $films = $filmModel->getAllFilms(); 

        // Ambil jumlah total untuk bagian "Counts" (menggunakan countAll() jika ada di model)
        // Ini lebih efisien daripada mengambil semua data lalu menghitung di PHP.
        $total_users = $userModel->countAll(); // Asumsi ada method countAll() di UsersModel
        $total_studios = $studioModel->countAll(); // Asumsi ada method countAll() di StudioModel
        $total_kategoris = $kategoriModel->countAll(); // Asumsi ada method countAll() di KategoriModel
        $total_films_count = $filmModel->countAll(); // Asumsi ada method countAll() di FilmModel

        $data = [
            'films' => $films, // Untuk bagian featured films
            'total_users' => $total_users,
            'total_studios' => $total_studios,
            'total_kategoris' => $total_kategoris,
            'total_films' => $total_films_count
        ];

        // Memuat views/home.php sebagai halaman utama
        $this->view('home', $data);
    }

    /**
     * Menampilkan detail film berdasarkan ID.
     * Dipanggil oleh index.php?hal=film_detail&id=X
     * @param int $film_id ID film.
     */
    public function detail($film_id) {
        $filmModel = new Film($this->conn);
        $jadwalModel = new Jadwal($this->conn);

        $film = $filmModel->getFilmById($film_id); // Mengambil detail film (sudah JOIN kategori & studio)
        $jadwal = $jadwalModel->getJadwalByFilmId($film_id); // Mengambil jadwal untuk film ini

        if (!$film) {
            $this->setErrorMessage("Film tidak ditemukan.");
            $this->redirect('film'); // Redirect ke daftar film jika film tidak ada
            exit();
        }

        $this->view('film_detail', ['film' => $film, 'jadwal' => $jadwal]);
    }

    /**
     * Menampilkan daftar semua film (film_list.php).
     * Dipanggil oleh index.php?hal=film
     */
    public function listFilms() {
        $filmModel = new Film($this->conn);
        $films = $filmModel->getAllFilms(); // Mengambil semua film (hanya ID, judul, studio_id)

        // Jika views/film_list.php membutuhkan detail kategori/studio, 
        // Anda mungkin perlu memanggil FilmModel->getAllFilmsWithDetails() jika ada
        // atau melakukan JOIN di FilmModel->getAllFilms() untuk itu.
        // Atau ambil film satu per satu dengan getFilmById di loop jika jumlah film sedikit.

        $this->view('film_list', ['films' => $films]);
    }

    /**
     * Menampilkan formulir untuk menambah film baru.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=film_create
     */
    public function create() {
        $this->requireAdmin();

        $kategoriModel = new Kategori($this->conn);
        $studioModel = new Studio($this->conn);

        $kategoris = $kategoriModel->getAllKategoris(); // Asumsi ada method getAllKategoris()
        $studios = $studioModel->getAllStudios(); // Asumsi ada method getAllStudios()

        $this->view('film_form', ['kategoris' => $kategoris, 'studios' => $studios]);
    }

    /**
     * Memproses data POST dari formulir penambahan film baru.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=film_store
     */
    public function store() {
        $this->requireAdmin();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // Filter dan validasi input
            $judul = trim($_POST['judul'] ?? '');
            $tanggal_rilis = $_POST['tanggal_rilis'] ?? '';
            $sinopsis = trim($_POST['sinopsis'] ?? '');
            $kategori_id = (int)($_POST['kategori_id'] ?? 0);
            $studio_id = (int)($_POST['studio_id'] ?? 0);
            $cover = null;

            if (empty($judul) || empty($tanggal_rilis) || empty($sinopsis) || empty($kategori_id) || empty($studio_id)) {
                $this->setErrorMessage("Semua kolom (Judul, Tanggal Rilis, Sinopsis, Kategori, Studio) harus diisi.");
                $this->redirect('film_create');
                exit();
            }

            // Validasi tanggal
            if (!DateTime::createFromFormat('Y-m-d', $tanggal_rilis)) {
                $this->setErrorMessage("Format tanggal rilis tidak valid.");
                $this->redirect('film_create');
                exit();
            }

            // Proses upload cover
            if (isset($_FILES['cover']) && $_FILES['cover']['error'] === UPLOAD_ERR_OK) {
                $upload_dir = dirname(__DIR__) . '/assets/img/film_covers/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0777, true); // Buat direktori jika belum ada
                }
                
                // Pastikan nama file unik dan aman
                $file_extension = pathinfo($_FILES['cover']['name'], PATHINFO_EXTENSION);
                $file_name = uniqid('cover_') . '.' . $file_extension;
                $target_file = $upload_dir . $file_name;

                // Validasi tipe file (opsional tapi direkomendasikan)
                $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
                if (!in_array($_FILES['cover']['type'], $allowed_types)) {
                    $this->setErrorMessage("Tipe file cover tidak diizinkan. Hanya JPG, PNG, GIF.");
                    $this->redirect('film_create');
                    exit();
                }

                if (move_uploaded_file($_FILES['cover']['tmp_name'], $target_file)) {
                    $cover = 'assets/img/film_covers/' . $file_name;
                } else {
                    $this->setErrorMessage("Gagal mengupload cover film. Periksa izin direktori.");
                    $this->redirect('film_create');
                    exit();
                }
            }

            $filmModel = new Film($this->conn);
            // Panggil createFilm (bukan addFilm) untuk konsistensi
            $result = $filmModel->createFilm($judul, $tanggal_rilis, $sinopsis, $cover, $kategori_id, $studio_id);

            if ($result) {
                $this->setSuccessMessage("Film berhasil ditambahkan.");
                $this->redirect('film'); // Redirect ke daftar film
                exit();
            } else {
                $this->setErrorMessage("Gagal menambahkan film. Pastikan tidak ada duplikasi data unik atau ada masalah database.");
                $this->redirect('film_create');
                exit();
            }
        } else {
            $this->redirect('film_create'); // Redirect jika bukan POST request
        }
    }

    /**
     * Menampilkan formulir untuk mengedit film yang sudah ada.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=film_edit&id=X
     * @param int $film_id ID film yang akan diedit.
     */
    public function edit($film_id) {
        $this->requireAdmin();

        $filmModel = new Film($this->conn);
        $kategoriModel = new Kategori($this->conn);
        $studioModel = new Studio($this->conn);

        $film = $filmModel->getFilmById($film_id); // Mengambil detail film
        $kategoris = $kategoriModel->getAllKategoris();
        $studios = $studioModel->getAllStudios();

        if (!$film) {
            $this->setErrorMessage("Film tidak ditemukan.");
            $this->redirect('film');
            exit();
        }

        $this->view('film_form_update', ['film' => $film, 'kategoris' => $kategoris, 'studios' => $studios]);
    }

    /**
     * Memproses data POST dari formulir pembaruan film.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=film_update
     */
    public function update() {
        $this->requireAdmin();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $id = $_POST['id'] ?? null;
            $judul = trim($_POST['judul'] ?? '');
            $tanggal_rilis = $_POST['tanggal_rilis'] ?? '';
            $sinopsis = trim($_POST['sinopsis'] ?? '');
            $kategori_id = (int)($_POST['kategori_id'] ?? 0);
            $studio_id = (int)($_POST['studio_id'] ?? 0);
            $old_cover = $_POST['old_cover'] ?? null; // Cover lama jika tidak ada upload baru

            $cover = $old_cover; // Default cover adalah cover lama

            // Validasi input
            if (empty($id) || empty($judul) || empty($tanggal_rilis) || empty($sinopsis) || empty($kategori_id) || empty($studio_id)) {
                $this->setErrorMessage("Data tidak lengkap untuk pembaruan film.");
                $this->redirect('film_edit', ['id' => $id]);
                exit();
            }
            if (!DateTime::createFromFormat('Y-m-d', $tanggal_rilis)) {
                $this->setErrorMessage("Format tanggal rilis tidak valid.");
                $this->redirect('film_edit', ['id' => $id]);
                exit();
            }

            // Proses upload cover baru
            if (isset($_FILES['cover']) && $_FILES['cover']['error'] === UPLOAD_ERR_OK) {
                $upload_dir = dirname(__DIR__) . '/assets/img/film_covers/';
                // Pastikan direktori ada
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0777, true);
                }
                
                $file_extension = pathinfo($_FILES['cover']['name'], PATHINFO_EXTENSION);
                $file_name = uniqid('cover_') . '_' . basename($_FILES['cover']['name']);
                $target_file = $upload_dir . $file_name;

                $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
                if (!in_array($_FILES['cover']['type'], $allowed_types)) {
                    $this->setErrorMessage("Tipe file cover tidak diizinkan. Hanya JPG, PNG, GIF.");
                    $this->redirect('film_edit', ['id' => $id]);
                    exit();
                }

                if (move_uploaded_file($_FILES['cover']['tmp_name'], $target_file)) {
                    $cover = 'assets/img/film_covers/' . $file_name;
                    // Hapus cover lama jika ada dan berhasil upload cover baru
                    if ($old_cover && file_exists(dirname(__DIR__) . '/' . $old_cover)) {
                        unlink(dirname(__DIR__) . '/' . $old_cover);
                    }
                } else {
                    $this->setErrorMessage("Gagal mengupload cover film baru. Periksa izin direktori.");
                    $this->redirect('film_edit', ['id' => $id]);
                    exit();
                }
            }

            $filmModel = new Film($this->conn);
            $result = $filmModel->updateFilm($id, $judul, $tanggal_rilis, $sinopsis, $cover, $kategori_id, $studio_id);

            if ($result) {
                $this->setSuccessMessage("Film berhasil diperbarui.");
                $this->redirect('film');
                exit();
            } else {
                $this->setErrorMessage("Gagal memperbarui film. Pastikan semua data benar atau ada masalah database.");
                $this->redirect('film_edit', ['id' => $id]);
                exit();
            }
        } else {
            $this->redirect('film'); // Redirect jika bukan POST request
        }
    }

    /**
     * Menghapus film dari database.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=film_destroy&id=X
     * @param int $film_id ID film yang akan dihapus.
     */
    public function destroy($film_id) {
        $this->requireAdmin();

        $filmModel = new Film($this->conn);
        $jadwalModel = new Jadwal($this->conn);
        $detailPemesananModel = new Detail_Pemesanan($this->conn);
        
        // 1. Cek jadwal terkait film ini
        $jadwal_film_ini = $jadwalModel->getJadwalByFilmId($film_id);
        $has_related_pemesanan = false;

        if (!empty($jadwal_film_ini)) {
            // 2. Jika ada jadwal, cek apakah ada pemesanan terkait jadwal-jadwal tersebut
            foreach($jadwal_film_ini as $j) {
                if ($detailPemesananModel->getPemesananCountByJadwal($j['id']) > 0) {
                    $has_related_pemesanan = true;
                    break; // Hentikan loop jika ditemukan pemesanan terkait
                }
            }
        }
        
        if ($has_related_pemesanan) {
            $this->setErrorMessage("Tidak bisa menghapus film. Ada jadwal tayang yang terkait dengan film ini dan sudah memiliki pemesanan. Harap batalkan/hapus pemesanan tersebut terlebih dahulu.");
            $this->redirect('film');
            exit();
        } else {
            // 3. Jika tidak ada pemesanan terkait jadwal, hapus semua jadwal film ini
            // Pastikan deleteJadwalByFilmId() di JadwalModel berfungsi dengan baik.
            $jadwalModel->deleteJadwalByFilmId($film_id);
            
            // 4. Ambil informasi film untuk menghapus cover file
            $film_to_delete = $filmModel->getFilmById($film_id);

            // 5. Baru hapus filmnya dari tabel
            $result = $filmModel->deleteFilm($film_id);

            if ($result) {
                // 6. Hapus file cover fisik jika ada
                if ($film_to_delete && $film_to_delete['cover'] && file_exists(dirname(__DIR__) . '/' . $film_to_delete['cover'])) {
                    unlink(dirname(__DIR__) . '/' . $film_to_delete['cover']);
                }
                $this->setSuccessMessage("Film berhasil dihapus beserta jadwal terkait.");
            } else {
                $this->setErrorMessage("Gagal menghapus film. Kendala database atau jadwal belum terhapus sempurna.");
            }
            $this->redirect('film');
            exit();
        }
    }
}